<?php

namespace App\Http\Controllers;

use App\Models\Contacto;
use App\Models\EntidadComercial;
use App\Models\Usuario;
use Carbon\Carbon;
use Exception;
use Illuminate\Auth\Access\AuthorizationException;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Str;

class EntidadComercialController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth:api');
    }
    public function crearEntidadComercial(Request $request)
    {
        try {
            // $this->authorize('registrar-entidadesComerciales');
            $validator = Validator::make($request->all(), [
                'ruc_empresa' => 'required|unique:entidades_comerciales,ruc_empresa',
                'nombre_completo' => 'required',
                'nombre_comercial' => 'required',
                'correo' => 'required',
                'direccion' => 'required',
                'logo' => 'required',
                'correo_coorporativo' => 'nullable',
                'usuario_registra' => 'required|exists:usuarios,id',
                'tipo_entidad' => 'required',

                'celular' => 'required',
                'fijo' => 'required',
                'whatsapp' => 'required'
            ]);
            if ($validator->fails()) {
                return response()->json([
                    'status' => 422,
                    'message' => 'Error al validar los datos de entrada.',
                    'data' => $validator->errors()
                ], 422);
            } else {
                $entidad_comercial = EntidadComercial::create([
                    'ruc_empresa' => $request->ruc_empresa,
                    'nombre_completo' => $request->nombre_completo,
                    'nombre_comercial' => $request->nombre_comercial,
                    'correo' => $request->correo,
                    'direccion' => $request->direccion,
                    'logo' => $request->logo,
                    'correo_coorporativo' => $request->correo_coorporativo,
                    'usuario_registra' => $request->usuario_registra,
                    'tipo_entidad' => $request->tipo_entidad,
                    'fecha_registro' => Carbon::now()->isoFormat('Y-M-DD')
                ]);
                $contacto = Contacto::create([
                    'celular' => $request->celular,
                    'fijo' => $request->fijo,
                    'whatsapp' => $request->whatsapp,
                    'entidad_id' => $entidad_comercial->id
                ]);

                //*****NOTA: CREAR POR DEFAUT INTERFAZ ENTIDAD CON COLORES DE FIRMAS EC*/







                return response()->json([
                    'status' => 201,
                    'message' => 'Entidad Comercial creada correctamente.',
                    'data' => $entidad_comercial
                ], 201);
            }
        } catch (AuthorizationException $th) {
            return response()->json([
                'status' => $th->getCode(),
                'message' => 'No autorizado!.',
                'data' => $th->getMessage()
            ], 401);
        } catch (Exception $e) {
            return response()->json([
                'status' => $e->getCode(),
                'message' => 'Ocurrio un error!.',
                'data' => $e->getMessage()
            ], 400);
        }
    }

    public function listarEntidadesComerciales(Request $request)
    {
        try {
            // $this->authorize('listar-entidadesComerciales');
            $request->merge([
                'page' => $request->input('page', 0) + 1
            ]);
            $lst_entidades_comerciales = EntidadComercial::with('contacto')
                ->where('estado', true)
                ->orderBy('updated_at', 'desc')
                ->paginate($request->query('per_page') ?? 10);

            foreach ($lst_entidades_comerciales as $key => $value) {
                $value['usuario_registra'] = Usuario::where('id', $value->usuario_registra)->first();
            }

            if ($lst_entidades_comerciales != null) {
                return response()->json([
                    'status' => 200,
                    'message' => 'Lista de Entidades Comerciales.',
                    'data' => $lst_entidades_comerciales
                ], 200);
            } else {
                return response()->json([
                    'status' => 200,
                    'message' => 'No hay Entidades Comerciales regitradas.',
                    'data' => null
                ], 200);
            }
        } catch (AuthorizationException $th) {
            return response()->json([
                'status' => $th->getCode(),
                'message' => 'No autorizado!.',
                'data' => $th->getMessage()
            ], 401);
        } catch (Exception $e) {
            return response()->json([
                'status' => $e->getCode(),
                'message' => 'Ocurrio un error!.',
                'data' => $e->getMessage()
            ], 400);
        }
    }

    public function actualizarEntidadComercial(Request $request, $id)
    {
        try {
            // $this->authorize('actualizar-entidadesComerciales');
            $entidad_comercial = EntidadComercial::with('contacto', 'interfaz')->where('id', $id)->first();
            if ($entidad_comercial != null) {
                $entidad_comercial->update($request->all());
                $entidad_comercial->contacto->update($request->all());
                $entidad_comercial->interfaz->update($request->all());
                return response()->json([
                    'status' => 200,
                    'message' => 'Entidad Comercial actualizada correctamente.',
                    'data' => $entidad_comercial
                ], 200);
            } else {
                return response()->json([
                    'status' => 200,
                    'message' => 'No se encontro la Entidad Comercial indicada.',
                    'data' => null
                ], 200);
            }
        } catch (AuthorizationException $th) {
            return response()->json([
                'status' => $th->getCode(),
                'message' => 'No autorizado!.',
                'data' => $th->getMessage()
            ], 401);
        } catch (Exception $e) {
            return response()->json([
                'status' => $e->getCode(),
                'message' => 'Ocurrio un error!.',
                'data' => $e->getMessage()
            ], 400);
        }
    }

    public function eliminarEntidadComercial($id)
    {
        try {
            // $this->authorize('eliminar-entidadesComerciales');
            $entidad = EntidadComercial::find($id);
            if ($entidad != null) {
                $entidad->update([
                    'estado' => false
                ]);
                return response()->json([
                    'status' => 200,
                    'message' => 'Entidad Comercial eliminada correctamente.',
                    'data' => null
                ], 200);
            } else {
                return response()->json([
                    'status' => 200,
                    'message' => 'No se encontro la Entidad Comercial indicada.',
                    'data' => null
                ], 200);
            }
        } catch (\Throwable $th) {
            return response()->json([
                'status' => $th->getCode(),
                'message' => 'Ocurrio un error!.',
                'data' => $th->getMessage()
            ], $th->getCode());
        }
    }

    public function verEntidadComercial($entidad)
    {
        try {
            // $this->authorize('ver-entidadesComerciales');
            if (Str::isUuid($entidad)) {
                $entidad_comercial = EntidadComercial::with('contacto', 'interfaz')->where('estado', true)->where('id', $entidad)->first();
                $entidad_comercial['usuario_registra'] = Usuario::where('id', $entidad_comercial->usuario_registra)->first();
                if ($entidad_comercial != null) {
                    return response()->json([
                        'status' => 200,
                        'message' => 'Datos de Entidad Comercial.',
                        'data' => $entidad_comercial
                    ], 200);
                } else {
                    return response()->json([
                        'status' => 200,
                        'message' => 'No se encontro datos de la Entidad Comercial indicada.',
                        'data' => null
                    ], 200);
                }
            } else {
                $entidad_comercial = EntidadComercial::with('contacto', 'interfaz')->where('estado', true)->where('ruc_empresa', $entidad)->first();
                if ($entidad_comercial != null) {
                    return response()->json([
                        'status' => 200,
                        'message' => 'Datos de Entidad Comercial.',
                        'data' => $entidad_comercial
                    ], 200);
                } else {
                    return response()->json([
                        'status' => 200,
                        'message' => 'No se encontro datos de la Entidad Comercial indicada.',
                        'data' => null
                    ], 200);
                }
            }
        } catch (\Throwable $th) {
            return response()->json([
                'status' => $th->getCode(),
                'message' => 'Ocurrio un error!.',
                'data' => $th->getMessage()
            ]);
        }
    }

    /*
    Permite actualizar la entidad comercial del usuario logueado
    ROL SOCIO_MASTER AND SOCIO_SUPERVISOR
    */
    public function actualizarMiEntidadComercial(Request $request)
    {
        try {
            // $this->authorize('actualizar-mi-entidadComercial');
            $usuario = Auth::user();
            $entidad_comercial = EntidadComercial::with('contacto', 'interfaz')->where('id', $usuario->entidad_id)->first();
            if ($entidad_comercial != null) {
                $entidad_comercial->update($request->all());
                $entidad_comercial->contacto->update($request->all());
                $entidad_comercial->interfaz->update($request->all());
                return response()->json([
                    'status' => 200,
                    'message' => 'Entidad Comercial actualizada correctamente.',
                    'data' => $entidad_comercial
                ], 200);
            } else {
                return response()->json([
                    'status' => 200,
                    'message' => 'No se encontro la Entidad Comercial indicada.',
                    'data' => null
                ], 200);
            }
        } catch (AuthorizationException $th) {
            return response()->json([
                'status' => $th->getCode(),
                'message' => 'No autorizado!.',
                'data' => $th->getMessage()
            ], 401);
        } catch (Exception $e) {
            return response()->json([
                'status' => $e->getCode(),
                'message' => 'Ocurrio un error!.',
                'data' => $e->getMessage()
            ], 400);
        }
    }

    /*
    Permite datos de mi entidad comercial
    ROL CUALQUIER USUARIO LOGUEADO
    */
    public function verMiEntidadComercial()
    {
        try {
            // $this->authorize('ver-mi-entidadComercial');
            $usuario = Auth::user();
            $entidad_comercial = EntidadComercial::with('contacto', 'interfaz')->where('id', $usuario->entidad_id)->first();
            if ($entidad_comercial != null) {
                return response()->json([
                    'status' => 200,
                    'message' => 'Datos de Entidad Comercial.',
                    'data' => $entidad_comercial
                ], 200);
            } else {
                return response()->json([
                    'status' => 200,
                    'message' => 'No se encontro datos de la Entidad Comercial indicada.',
                    'data' => null
                ], 200);
            }
            // } catch (AuthorizationException $th) {
            //     return response()->json([
            //         'status'=>$th->getCode(),
            //         'message'=>'No autorizado!.',
            //         'data'=>$th->getMessage()
            //     ],401);
        } catch (Exception $e) {
            return response()->json([
                'status' => $e->getCode(),
                'message' => 'Ocurrio un error!.',
                'data' => $e->getMessage()
            ], 400);
        }
    }
    public function listarTodosUsuarios(Request $request)
    {
        try {
            $request->merge([
                'page' => $request->input('page', 0) + 1
            ]);
            $lst_usuarios = Usuario::with('tipo_usuario', 'cod_pais', 'entidad_id', 'contacto')->where('estado', true)->paginate($request->query('per_page') ?? 10);
            if ($lst_usuarios != null) {
                return response()->json([
                    'status' => 200,
                    'message' => 'Lista de  todos los Usuarios.',
                    'data' => $lst_usuarios
                ], 200);
            } else {
                return response()->json([
                    'status' => 200,
                    'message' => 'No hay Usuarios registrados.',
                    'data' => null
                ], 200);
            }
        } catch (\Throwable $th) {
            return response()->json([
                'status' => $th->getCode(),
                'message' => 'Ocurrio un error!.',
                'data' => $th->getMessage()
            ], $th->getCode());
        }
    }

    public function listarUsuariosEntidadComercial($entidad)
    {
        try {
            $lst_usuarios = Usuario::with('tipo_usuario')->where('entidad_id', $entidad)->get();
            if ($lst_usuarios != null) {
                return response()->json([
                    'status' => 200,
                    'message' => 'Lista de usuarios de Entidad Comercial indicada. ',
                    'data' => $lst_usuarios
                ]);
            } else {
                return response()->json([
                    'status' => 200,
                    'message' => 'No existe usuario en la Entidad Comercial. ',
                    'data' => $lst_usuarios
                ]);
            }
        } catch (\Throwable $th) {
            return response()->json([
                'status' => $th->getCode(),
                'message' => 'Ocurrio un error!. ',
                'data' => $th->getMessage()
            ], $th->getCode());
        }
    }

    public function buscarEntidadRuc(Request $request)
    {
        try {
            // $this->authorize('buscar-entidad-ruc');
            $entidad = EntidadComercial::with('usuario')->where('ruc_empresa', $request->ruc_empresa)->first();
            if ($entidad != null) {
                return response()->json([
                    'status' => 200,
                    'message' => 'Entidad Comercial encontrada.',
                    'data' => $entidad
                ], 200);
            } else {
                return response()->json([
                    'status' => 404,
                    'message' => 'No se encontro la Entidad Comercial indicada.',
                    'data' => null
                ], 404);
            }
        } catch (AuthorizationException $th) {
            return response()->json([
                'status' => $th->getCode(),
                'message' => 'No autorizado!.',
                'data' => $th->getMessage()
            ], 401);
        } catch (Exception $e) {
            return response()->json([
                'status' => $e->getCode(),
                'message' => 'Ocurrio un error!.',
                'data' => $e->getMessage()
            ], 400);
        }
    }
}
