<?php

namespace App\Http\Controllers;

use App\Interfaces\TipoFirmaRepositoryInterface;
use App\Models\TipoFirma;
use App\Repositories\TipoFirmaRepository;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use App\Models\ApiCoorporativo;
use App\Models\PrecioFirma;
use Carbon\Carbon;


class TipoFirmaPublicController extends Controller
{
    private $tipofirma;

    public function __construct(TipoFirmaRepository $tipofirma) 
    {
        $this->tipofirma = $tipofirma;
        // $this->middleware('auth');
    }

    public function listarTiposFirmas1(Request $request)
    {
        try {
            $validator=Validator::make($request->all(),[
                'token_api'=>'required|exists:api_coorporativos,token',
                'clave_api'=>'required|exists:api_coorporativos,clave_coorporativo'
            ]);
            if ($validator->fails()) {
                return response()->json([
                    'status'=>404,
                    'message'=>'Error al validar los datos de entrada.',
                    'data'=>$validator->errors()
                ],404);
            } else {
                return ($this->tipofirma->getAllTiposFirmas() != null) ?
                response()->json([
                    'status' => 200,
                    'message' => 'Listado de Tipos de Firmas.',
                    'data' => $this->tipofirma->getAllTiposFirmas()
                ]) :
                response()->json([
                    'status' => 200,
                    'message' => 'No hay Tipos de Firmas Registradas.',
                    'data' => null
                ]);
            } 

        } catch (\Throwable $th) {
            response()->json([
                'status' => $th->getCode(),
                'message' => 'Ocurrio un error!.',
                'data' => $th->getMessage()
            ],400);
        }
    }
    
    public function listarTiposFirmas(Request $request)
    {
        try {
            $token = $request->input('token_api');
            $clave = $request->input('clave_api');
    
            $api = \App\Models\ApiCoorporativo::where('token', $token)
                ->where('clave_coorporativo', $clave)
                ->where('estado', 1)
                ->first();
    
            if (!$api) {
                return response()->json([
                    'code' => 401,
                    'message' => 'Token y/o Clave Incorrectos',
                    'data' => null
                ], 401);
            }
    
            $entidad_id = $api->entidad_id;
    
            $tiposFirmas = \App\Models\TipoFirma::orderBy('tiempo_validez', 'asc')->get();
    
         $tiposFirmasConPrecio = $tiposFirmas->map(function ($firma) use ($entidad_id) {
                $precio = \App\Models\PrecioFirma::where('entidad_id', $entidad_id)
                    ->where('tipo_firma_id', $firma->id)
                    ->orderBy('created_at', 'desc')
                    ->first();
            
                if ($precio) {
                    $pvp = floatval($precio->pvp);
                    $subtotal = round($pvp / 1.15, 2);
                    $iva = round($subtotal * 0.15, 2);
                    $retencionRenta = round($subtotal * 0.01, 2);
                    $retencionIva = round($iva * 0.70, 2);
            
                    return [
                        'id' => $firma->id,
                        'tipo_firma' => $firma->tipo_firma,
                        'descripcion' => $firma->descripcion,
                        'tiempo_validez' => $firma->tiempo_validez,
                        'precio' => number_format($pvp, 2, '.', ''),
                        'subtotal' => number_format($subtotal, 2, '.', ''),
                        'iva' => number_format($iva, 2, '.', ''),
                        'retencionRenta' => number_format($retencionRenta, 2, '.', ''),
                        'retencionIva' => number_format($retencionIva, 2, '.', ''),
                    ];
                }
            
                return null;
            })->filter()->values(); // Filtra los nulls
    
            return response()->json([
                'status' => 200,
                'message' => 'Listado de Tipos de Firmas.',
                'data' => $tiposFirmasConPrecio->values() // Reindexa el array
            ]);
        } catch (\Throwable $th) {
            return response()->json([
                'status' => 500,
                'message' => 'Ocurri un error.',
                'data' => $th->getMessage()
            ], 500);
        }
    }
    
    public function consultarPrecio(Request $request)
    {
        $token = $request->input('token_api');
        $clave = $request->input('clave_api');
        $tipoFirmaNombre = $request->input('tipo_firma'); // No se usa, solo se devuelve
        $vigencia = $request->input('vigencia');
    
        // Buscar entidad por token y clave
        $apiCorporativo = \App\Models\ApiCoorporativo::where('token', $token)
            ->where('clave_coorporativo', $clave)
            ->first();
    
        if (!$apiCorporativo) {
            return response()->json([
                'status' => 401,
                'message' => 'Token o clave invlidos.',
                'data' => []
            ]);
        }
    
        $entidad_id = $apiCorporativo->entidad_id;
    
        // Buscar tipo_firma "ARCHIVO"
        $tipoArchivo = \App\Models\TipoFirma::where('tipo_firma', 'ARCHIVO')->first();
    
        if (!$tipoArchivo) {
            return response()->json([
                'status' => 404,
                'message' => 'Tipo de firma ARCHIVO no encontrado.',
                'data' => []
            ]);
        }
    
        // Obtener precio vigente para la entidad y tipo ARCHIVO
        $precio = \App\Models\PrecioFirma::where('entidad_id', $entidad_id)
            ->where('tipo_firma_id', $tipoArchivo->id)
            ->where('estado', 1)
            ->orderBy('created_at', 'desc')
            ->first();
    
        if (!$precio || $precio->pvp === null) {
            return response()->json([
                'status' => 404,
                'message' => 'No se encontr un precio vigente para esta entidad.',
                'data' => []
            ]);
        }
    
        $pvp = floatval($precio->pvp);
        $subtotal = round($pvp / 1.15, 2);
        $iva = round($subtotal * 0.15, 2);
        $retencionRenta = round($subtotal * 0.01, 2);
        $retencionIva = round($iva * 0.70, 2);
    
        return response()->json([
            'status' => 200,
            'message' => 'Precio consultado correctamente.',
            'data' => [
                'tipo_firma' => $tipoFirmaNombre,
                'vigencia' => $vigencia,
                'precio' => number_format($pvp, 2, '.', ''),
                'subtotal' => number_format($subtotal, 2, '.', ''),
                'iva' => number_format($iva, 2, '.', ''),
                'retencionRenta' => number_format($retencionRenta, 2, '.', ''),
                'retencionIva' => number_format($retencionIva, 2, '.', ''),
            ]
        ]);
    }


    public function listarTiposFirmas2(Request $request)
    {
        try {
            $token = $request->input('token_api');
            $clave = $request->input('clave_api');
    
            // Buscar entidad por token y clave
            $api = ApiCoorporativo::where('token', $token)
                ->where('clave_coorporativo', $clave)
                ->where('estado', 1)
                ->first();
    
            if (!$api) {
                return response()->json([
                    'code' => 401,
                    'message' => 'Token y/o Clave Incorrectos',
                    'data' => null
                ], 401);
            }
    
            $entidad_id = $api->entidad_id;
    
            // Obtener tipos de firma
            $tiposFirmas = TipoFirma::orderBy('tiempo_validez', 'asc')->get();
    
            // Para cada tipo de firma, buscar precio vigente
            $fechaActual = Carbon::now();
    
            $tiposFirmasConPrecio = $tiposFirmas->map(function ($firma) use ($entidad_id, $fechaActual) {
                $precio = PrecioFirma::where('entidad_id', $entidad_id)
                    ->where('tipo_firma_id', $firma->id)
                    ->where('estado', 1)
                    ->whereDate('fecha_inicio', '<=', $fechaActual)
                    ->whereDate('fecha_fin', '>=', $fechaActual)
                    ->orderBy('fecha_inicio', 'desc')
                    ->first();
    
                $firma->precio = $precio ? $precio->precio : null;
                $firma->pvp = $precio ? $precio->pvp : null;
    
                return $firma;
            });
    
            return response()->json([
                'status' => 200,
                'message' => 'Listado de Tipos de Firmas.',
                'data' => $tiposFirmasConPrecio
            ]);
        } catch (\Throwable $th) {
            return response()->json([
                'status' => $th->getCode(),
                'message' => 'Ocurrió un error.',
                'data' => $th->getMessage()
            ], 400);
        }
    }
}
